/* -*-C-*-
 ##############################################################################
 #
 # File:        demo/fft.c
 # RCS:         $Id: fft.c,v 1.9 1998/07/20 22:59:24 ericb Exp $
 # Description: Host program for doing inline FFTs
 # Created:     March 20, 1996
 # Language:    C
 # Package:     E1432
 #
 # Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # This program requires an E1432 or E1433 at logical address 8 with a
 # 1 volt peak input signal connected to channel 1 for correct triggering.
 #
 # Revisions:
 #
 #
 ##############################################################################
 */
#include <math.h>
#include <stdlib.h>		/* For exit */
#include <stdio.h>		/* For printf */
#include <unistd.h>		/* For sleep */
#include "e1432.h"
#include "xplot.h"
#include "err1432.h"

/*#define TRAILER  */
/*#define FREQ_ONLY*/

#define LA		8

#define WIDTH           140
#define HEIGHT          80 
#define NUM_CHANS	16

#define RANGE		1.0	
#define BLOCK_SIZE	1024
#define SPAN		20000
#define AVG_MODE	E1432_AVG_NONE
#define AVG_NUMBER	10
#define AVG_UPDATE	1
#define AVG_WEIGHT	5.0

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

int
main(void)
{
    int  i, j, k, nchan;
    LONGSIZ32 count;
    SHORTSIZ16 la = LA;
    SHORTSIZ16 chan_list[NUM_CHANS];
    SHORTSIZ16 inputs;
    E1432ID hw;
    struct e1432_hwconfig cf;
    SHORTSIZ16 error;
    SHORTSIZ16 avgMode;
    FLOATSIZ32 *data[NUM_CHANS];
    FLOATSIZ32 *freq[NUM_CHANS], *freqBuf, sum;
    long points = BLOCK_SIZE;
    long freq_points, freq_blocksize;
    char *plotid[32];
    int row, col, id;  
    char geometry[80];
    char title[80];
    float temp, floatRange;
    struct e1432_trailer trailer;
    char *semabinloc = "/opt/e1432/lib/sema.bin";

    for(i=0; i < NUM_CHANS; i++)
    {
	data[i] = NULL;
	freq[i] = NULL;
	plotid[i] = NULL;
	plotid[i + NUM_CHANS] = NULL;
    }

    avgMode = AVG_MODE;

    if(avgMode == E1432_AVG_NONE || avgMode == E1432_AVG_LIN)  
	freq_blocksize = BLOCK_SIZE;
    else
	freq_blocksize = BLOCK_SIZE / 2;

    freq_points = (int)((float)points / 2.56) + 1;
    
    temp = (float)(points - 1);

    if(e1432_init_io_driver()) {
       (void) printf("e1432_init_io_driver() failed\n");
       exit(0);
    }

    CHECK(e1432_print_errors(0));

    /* install the downloadable code */
    (void) printf("Checking for E1432 with firmware at logical address %d ... ",
									la);
    (void) fflush(stdout);
    error = e1432_get_hwconfig(1, &la, &cf);
    CHECK(e1432_print_errors(1));
    if (error)
    {
	(void) printf("Not found.\n\n");
	(void) printf("Installing firmware from %s into E1432 at la %d ... ",
					   semabinloc, la);
	(void) fflush(stdout);
	error = e1432_install(1, &la, 0, semabinloc);
	if (error)
	{
	    (void) printf("\ne1432_install failed and returned error %s\n");
	    exit(0);
	}
	(void) printf("Done.\n");
    }
    else
    {
	(void) printf("Found.\n");
    }
    (void) printf("\n");

    CHECK(e1432_assign_channel_numbers(1, &la, &hw));

    /* Create channel group */
    error = e1432_get_hwconfig(1, &la, &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(0);
    }
    nchan = cf.input_chans;

    (void) printf("Found %d input channels\n", nchan);

    /* Create channel group */
    for(i=0; i<nchan; i++)
       chan_list[i] = E1432_INPUT_CHAN(i+1);

    inputs = e1432_create_channel_group(hw, nchan, chan_list);
    if (inputs >= 0)
    {
	(void) printf("e1432_create_channel_group inputs returned %d\n",
			    inputs);
	return -1;
    }

#if 0
    CHECK(e1432_set_internal_debug(hw, inputs, 0x400)); 
#endif

    /* Initialize hardware and measurment things */
    CHECK(e1432_set_analog_input(hw, inputs,
				     E1432_INPUT_MODE_VOLT,
				     E1432_INPUT_HIGH_NORMAL,
				     E1432_ANTI_ALIAS_ANALOG_ON,
				     E1432_COUPLING_DC, 
				     RANGE));

    CHECK(e1432_get_range(hw, chan_list[0], &floatRange));
    CHECK(e1432_set_data_size(hw, inputs, E1432_DATA_SIZE_16));
    CHECK(e1432_set_avg_mode(hw, inputs, AVG_MODE));
    CHECK(e1432_set_avg_number(hw, inputs, AVG_NUMBER));
    CHECK(e1432_set_avg_update(hw, inputs, AVG_UPDATE));
    CHECK(e1432_set_avg_weight(hw, inputs, AVG_WEIGHT));
    CHECK(e1432_set_blocksize(hw, inputs, BLOCK_SIZE)); 
    CHECK(e1432_set_data_mode(hw, inputs, E1432_BLOCK_MODE));
    CHECK(e1432_set_span(hw, inputs, SPAN));
    CHECK(e1432_set_window(hw, inputs, E1432_WINDOW_HANNING));
    CHECK(e1432_set_calc_data(hw, inputs, E1432_DATA_FREQ));  
    CHECK(e1432_set_trigger_channel(hw, chan_list[0], E1432_CHANNEL_ON));
    CHECK(e1432_set_trigger_mode(hw, inputs, E1432_TRIGGER_MODE_LEVEL));
    CHECK(e1432_set_auto_trigger(hw, inputs, E1432_MANUAL_TRIGGER));
    CHECK(e1432_set_trigger_slope(hw, inputs, E1432_TRIGGER_SLOPE_POS));
    CHECK(e1432_set_trigger_level(hw, inputs, E1432_TRIGGER_LEVEL_LOWER, 0.0));
    CHECK(e1432_set_trigger_level(hw, inputs, E1432_TRIGGER_LEVEL_UPPER, 0.0));

    /* set up xplot displays */
    row = col = 0;
    id = 0;

    for(i=0; i < nchan; i++) {
      if(data[i] == NULL) {
        data[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!data[i]) {
          (void) printf("Can't malloc data array of %d points\n", points);
          exit(0);
        }
	for(j=0; j < points; j++) {
	   data[i][j] = 0.0;
	}
      }

      if(freq[i] == NULL) {
        freq[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!freq[i]) {
          (void) printf("Can't malloc freq array of %d points\n", points);
          exit(0);
        }
	for(j=0; j < points; j++) {
	   freq[i][j] = 0.0;
	}
      }

      if(plotid[id] == NULL) { 
        (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
				(WIDTH + 20) * col, (HEIGHT + 40) * row ); 
	(void) sprintf(title, "Time %d", i + 1);

        plotid[id] = xplot_init_plot(data[i], points, temp, floatRange, 
			-floatRange, GENERIC_TRACE, geometry, title);
        (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
			(WIDTH + 20) * col, (HEIGHT + 40) * (row  + 1)); 
	(void) sprintf(title, "Freq %d", i + 1);

        plotid[id+8] = xplot_init_plot(freq[i], freq_points, SPAN, 0.0, 
			-100.0, FREQ_LOG_MAG_TRACE, geometry, title);
      }
      xplot_change_yautoscale(plotid[id], 0);
      xplot_change_yautoscale(plotid[id+8], 0);
      xplot_set_xscale(plotid[id], (float)0.0, temp);
      xplot_set_xscale(plotid[id+8], (float)0.0, SPAN);
      xplot_set_yscale(plotid[id], floatRange, -floatRange);
      xplot_set_yscale(plotid[id+8], 10.0, -100.0);
      xplot_change_xlabel(plotid[id], "Samples");
      xplot_change_xlabel(plotid[id+8], "Hz");
      xplot_change_ylabel(plotid[id], "Volts");
      xplot_change_ylabel(plotid[id+8], "dBV");
      xplot_repaint(plotid[id]); 
      xplot_repaint(plotid[id+8]); 
        
      col++;
      id++;
      if((col % 8) == 0) {
          col = 0;
          row +=2;
          id += 8;
      }

    }

#ifdef TRAILER
    CHECK(e1432_set_append_status(hw, inputs, E1432_APPEND_STATUS_ON));
#endif

#ifdef FREQ_ONLY
	CHECK(e1432_set_enable(hw, inputs, 
				E1432_ENABLE_TYPE_TIME, E1432_ENABLE_OFF));
#endif 

    /* Start measurement */
    CHECK(e1432_init_measure(hw, inputs));

    (void) sleep(1);

    for (;;)
    {
	/* Wait for block available */
	while(e1432_block_available(hw, inputs) <= 0)
		; 

#ifndef FREQ_ONLY
	for(i=0; i < nchan; i++) {	/* read time data */
            error = e1432_read_float32_data(hw, chan_list[i],
				E1432_TIME_DATA, data[i], points, 
				&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
            }

	}
#endif

	id = 0;
        for(i=0; i < nchan; i++) {	/* get frequency data */
	    freqBuf = freq[i];
	    error = e1432_read_float32_data(hw, chan_list[i], 
			E1432_FREQ_DATA, freqBuf, freq_blocksize, 
			&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
            }
	
	    /* make the 0 dB point full scale */
	    for(j=0; j < freq_blocksize; j++)
		freqBuf[j] /= floatRange;

	    if(avgMode == E1432_AVG_NONE || avgMode == E1432_AVG_LIN)  
	    {
		/* do a magnitude squared of all of the complex points */
		for(j=0,k=0; j < freq_points; j++)
		{
		    sum = freqBuf[k] * freqBuf[k];  /* real part */
		    k++;
	            
		    /* special treatment of dc point since the Fs/2 point
		       is uploaded as the imaginary part of the DC bin.  
		       Therefore the mag squared of the first point is
		       just the square of the real part, or the DC value */
		    if(j != 0)
		        sum += freqBuf[k] * freqBuf[k]; /* imag part */
		    k++;
		    freqBuf[j] = sum;
		}
	    }

	    /* convert to dBV */
	    for(j=0; j < freq_points; j++)
	    {
		if(freqBuf[j] < 1e-23)	/* limit really small values */
		{
		    freqBuf[j] = 10.0 * log10(1e-23);
		}
		else
		{
		    freqBuf[j] = 10.0 * log10(freqBuf[j]);
		}
	    }

#ifndef FREQ_ONLY
            xplot_check_events(plotid[id]);
            xplot_data_update(plotid[id]);
#endif
            xplot_check_events(plotid[id+8]);
            xplot_data_update(plotid[id+8]);    
	    if((++id%8)==0) id += 8;
        }

    }
    /*NOTREACHED*/
    return 0;
}

